;**********************************************************************
;                                                                     *
;    Filename:	    KS.asm                                            *
;    Date:          06.07.2006                                        *
;    File Version:  V1.3                                              *
;                                                                     *
;    Author:        Christian Batz                                    *
;                                                                     *
;                                                                     *
;**********************************************************************
;                                                                     *
;    Files required:                                                  *
;    12F629.lkr                                                       *
;                                                                     *
;                                                                     *
;**********************************************************************
;                                                                     *
;    Notes:                                                           *
;    If there are two claps the relais is switched and the LED is     *
;    on for 2 seconds. For every detected clap LED is on for 200ms .  *
;    The relais can also switched manually by using the switch.       *
;    If there are more than 4 relais switches by claps the LED is     *
;    flashing and you have to wait at least 60 seconds before the     *
;    next clap is accepted. Manual switch still works.                *
;    The current state of the relais is saved in the EEPROM.          *
;                                                                     *
;    Clap sequence:                                                   *
;    1st clap: max 100ms                                              *
;    Pause between claps: min 400ms                                   *
;    2nd clap wait: max 1s                                            *
;    2nd clap: max 100ms                                              *
;    No clap: min 400ms                                               *
;                                                                     *
;    After relais switch there is a break of 2 sconds.                *
;                                                                     *
;**********************************************************************

	list      p=12F629          ; list directive to define processor
	#include <p12F629.inc>      ; processor specific variable definitions

	errorlevel  -302            ; suppress message 302 from list file

	__CONFIG   _CP_OFF & _CPD_OFF & _BODEN_OFF & _MCLRE_ON & _WDT_OFF & _PWRTE_ON & _INTRC_OSC_NOCLKOUT


;***** VARIABLE DEFINITIONS

INT_VAR		UDATA_SHR	0x20   
w_temp		RES		1			; variable used for context saving 
status_temp	RES     	1			; variable used for context saving
general		RES		1			; general flags
LEDH		RES		1			; postscaler LED
LEDH_DEF	RES		1			; default postscaler LED
LEDL		RES		1			; postscaler LED
Ton		RES		1			; counter for TMR0 activation
CSIG20		RES		1			; clap signal postscaler
SW4		RES		1			; switch signal postscaler
ML125		RES		1			; main loop postscaler
MLX		RES		1			; main loop postscaler
SCount		RES		1			; switch counter
SWC30		RES		1			; switch counter postscaler
SWC200		RES		1			; switch counter postscaler


;***** DEFINITIONS

#define		csig		general,0	; clap signal flag
#define		csig_w		general,1	; clap signal wait flag
#define		swsig		general,2	; manual switch flag
#define		swsig_w		general,3	; manual switch wait flag
#define		led_w		general,4	; LED wait flag
#define		sco_w		general,5	; switch counter wait flag
#define		led_flash	general,6	; LED flash flag

#define		RELAIS	GPIO,GP4		; relais port
#define		LED	GPIO,GP5		; LED port

;**********************************************************************
RESET_VECTOR	CODE	0x000			; processor reset vector
		goto    Main            	; go to beginning of program

;***** IRQ HANDLING
INT_VECTOR	CODE	0x004			; interrupt vector location
		movwf   w_temp          	; save off current W register contents
		movfw	STATUS			; move status register into W register
		movwf	status_temp     	; save off contents of STATUS register
		
;** INT HANDLING (COMPARATOR/INT/TMR1)

		bcf     STATUS,RP0     		; set file register bank to 0
		btfsc	PIR1,CMIF		; comparator int
		call 	Signal
		btfsc	INTCON,INTF		; external int
		call	Switch
		btfsc	PIR1,TMR1IF		; TMR1 overflow int
		call	Timing
		
		movfw	status_temp		; retrieve copy of STATUS register
		movwf	STATUS          	; restore pre-isr STATUS register contents
		swapf   w_temp,f
		swapf   w_temp,w        	; restore pre-isr W register contents
		retfie                  	; return from interrupt

;***** COMPARATOR IRQ
Signal		bcf	PIR1,CMIF		; clears comparator int flag
		bsf	STATUS,RP0		; bank 1
		btfss	PIE1,CMIE		; int allowed ?
		goto	NoSig
		bcf	STATUS,RP0		; bank 0
		
		btfsc	led_flash		; LED flashing ?
		return

		btfss	CMCON,COUT		; Vin+ > Vin-
		return
		
		bsf	STATUS,RP0		; bank 1
		bcf	PIE1,CMIE		; disables comparator int
		bcf	STATUS,RP0		; bank 0
		bsf	csig			; sets clap flag
		bsf	csig_w			; sets wait flag for clap
		movlw	d'20'			; sets clap signal postscaler
		movwf	CSIG20
		incf	Ton,f			; increases TMR1 on counter
		bsf	T1CON,TMR1ON		; starts TMR1

;** CLAP LED

		movlw	d'1'
		movwf	LEDL			; LED is on for 200ms
		movlw	d'40'			; sets LED postscaler
		movwf	LEDH
		movwf	LEDH_DEF		; sets LED postscaler default
		btfsc	led_w			; LED already on ?
		return
		bsf	led_w			; sets led wait flag
		bsf	LED			; enables LED
		incf	Ton,f			; increases TMR1 on counter
		return

;** NO COMPARATOR IRQ

NoSig		bcf	STATUS,RP0		; bank 0
		return

;***** EXTERNAL INT IRQ
Switch		bcf	INTCON,INTF		; clears external int flag
		btfss	INTCON,INTE		; int allowed ?
		return

		bcf	INTCON,INTE		; disables external int
		
		bsf     STATUS,RP0      	; set file register bank to 1
		btfss	OPTION_REG,INTEDG
		bsf	swsig			; sets switch flag
		movlw	b'01000000'
		xorwf	OPTION_REG,f		; changes int edge -> no action for rising edge
		bcf     STATUS,RP0      	; set file register bank to 0

		bsf	swsig_w			; sets wait flag for switch
		movlw	d'4'			; sets switch signal postscaler
		movwf	SW4
		bsf	T1CON,TMR1ON		; starts TMR1
		incf	Ton,f			; increases TMR1 on counter
		return

;***** TMR1 IRQ
Timing		bcf	PIR1,TMR1IF		; clears TMR1 overflow flag
		clrf	TMR1L			; clears low byte of TMR1
		movlw	0xec
		movwf	TMR1H			; setting hih byte of TMR1
		movlw	0x78
		movwf	TMR1L			; setting low byte of TMR1

		bsf     STATUS,RP0      	; set file register bank to 1
		btfss	PIE1,TMR1IE		; int allowed ?
		goto	NoTMR
		bcf     STATUS,RP0     		; set file register bank to 0
		
;** CLAP SIGNAL HANDLING

		btfss	csig_w			; clap sig wait
		goto	SWSig
		decfsz	CSIG20,f		; decreases postscaler -> waits 100ms
		goto	SWSig
		bcf	csig_w			; clears wait flag for clap
		bsf	STATUS,RP0		; bank 1
		bsf	PIE1,CMIE		; enables comparator int
		bcf	STATUS,RP0		; bank 0
		decf	Ton,f			; decreases TMR1 on counter

;** EXTERNAL SWITCH HANDLING

SWSig		btfss	swsig_w			; switch sig wait
		goto	LEDSig
		decfsz	SW4,f			; decreases postscaler -> waits 20ms
		goto	LEDSig
		bcf	swsig_w			; clears wait flag for switch
		bsf	INTCON,INTE		; enables external int
		decf	Ton,f			; decreases TMR1 on counter

;** LED HANDLING (ON/OFF/FLASH)

LEDSig		btfss	led_w			; LED on wait
		goto	SWCSig
		decfsz	LEDH,f			; decreases first postscaler -> waits 200ms, 2s or 60s flashing
		goto	SWCSig
		movfw	LEDH_DEF		; resets first postscaler
		movwf	LEDH
		btfss	led_flash		; flashing ?
		goto	LEDNorm
		movlw	b'100000'		; toggles LED
		xorwf	GPIO,f
LEDNorm		decfsz	LEDL,f			; finished ?
		goto	SWCSig
		bcf	LED			; disables LED
		bcf	led_w			; clears LED wait flag
		bcf	led_flash		; clears LED flashing flag
		decf	Ton,f			; decreases TMR1 on counter

;** CLAP SWITCH COUNTER HANDLING

SWCSig		btfss	sco_w			; switch counter wait
		goto	Tcheck
		decfsz	SWC200,f		; decreases first postscaler -> waits 30s
		goto	Tcheck
		movlw	d'200'			; resets first postscaler
		movwf	SWC200
		decfsz	SWC30,f			; finished ?
		goto	Tcheck
		decfsz	SCount,f		; decreases SCount
		goto	SWCRl
		bcf	sco_w			; clears switch counter wait flag
		decf	Ton,f			; decreases TMR1 on counter
		goto	Tcheck
SWCRl		movlw	d'30'			; resets scond postscaler
		movwf	SWC30

;** TMR1 STILL NEEDED ?

Tcheck		movfw	Ton			; TMR1 int still needed ?
		btfss	STATUS,Z
		return
		bcf	T1CON,TMR1ON		; stops TMR1
		movlw	0xec
		movwf	TMR1H			; setting high byte of TMR1
		movlw	0x78
		movwf	TMR1L			; setting low byte of TMR1
		return

;** NO TMR1 INT

NoTMR		bcf     STATUS,RP0		; set file register bank to 0
		return

;**** INIT
Init		call    0x3FF			; retrieve factory calibration value
		bsf     STATUS,RP0		; set file register bank to 1 
		movwf   OSCCAL			; update register with factory cal value
		movlw	b'00000111'		; TMR0 prescaling 256
		movwf	OPTION_REG
		movlw	b'001111'		; GP0 - GP3 are inputs, GP4 - GP5 are outputs
		movwf	TRISIO
		movlw	b'01010000'		; peripheral and INT irq enabled
		movwf	INTCON
		movlw	b'00001001'		; comparator and TMR1 int enabled
		movwf	PIE1
		clrf	WPU			; disables weak pull-ups
		clrf	EEADR			; adress is 0x00
		bsf	EECON1,RD		; starts reading
		movfw	EEDATA			; moves data into W
		bcf     STATUS,RP0      	; set file register bank to 0
		movwf	GPIO			; sets up ports
		movlw	b'00000010'		; sets up comparator
		movwf	CMCON
		clrf	general
		clrf	Ton
		clrf	SCount			; resets switch counter
		movlw	0xec			; TMR1 overflow after 5ms
		movwf	TMR1H			; setting high byte of TMR1
		movlw	0x78
		movwf	TMR1L			; setting low byte of TMR1
		retfie

;***** RELAIS (EEPROM) AND LED

Relais		bsf	STATUS,RP0		; bank 1
		bcf	PIE1,CMIE		; disables comparator int (no LED change)
		bcf	STATUS,RP0		; bank 0
		
;** RELAIS SWITCH

		movlw	b'010000'
		xorwf	GPIO,f			; toggles relais

;** SAVING TO EEPROM

		andwf	GPIO,w			; only relais bit
		bsf	STATUS,RP0		; bank 1
		movwf	EEDATA
		clrf	EEADR			; adress is 0x00
		bsf	EECON1,WREN		; enables write
		bcf	INTCON,GIE		; disables global ints
		movlw	0x55			; unlock write
		movwf	EECON2
		movlw	0xaa
		movwf	EECON2
		bsf	EECON1,WR		; starts writing
		bsf	INTCON,GIE		; enables global ints
EEWait		btfsc	EECON1,WR		; finished ?
		goto	EEWait
		bcf     STATUS,RP0		; bank 0

;** LED SETUP

		bcf	T1CON,TMR1ON		; stops TMR1
		movlw	d'2'			; sets postscalers
		movwf	LEDL			; LED is on for 2s
		movlw	d'200'			; sets LED postscaler
		movwf	LEDH
		movwf	LEDH_DEF		; sets LED postscaler default
		bcf	led_flash		; disables flashing
		btfsc	led_w			; LED already on ?
		goto	Ron
		bsf	LED			; enables LED
		bsf	led_w			; sets led wait flag
		incf	Ton,f			; increases TMR1 on counter
Ron		bsf	T1CON,TMR1ON		; starts TMR1

;** BREAK

		movlw	d'60'			; sets postscaler for break
		movwf	MLX
		movfw	TMR0
		addlw	d'125'			; sets postscaler for break
		movwf	ML125
Rpause		movfw	ML125			; waits 2s
		subwf	TMR0,w
		btfss	STATUS,Z
		goto	Rpause
		movlw	d'125'
		addwf	ML125,f
		decfsz	MLX,f
		goto	Rpause

;** SETUP FOR NEW INPUT

		bsf	STATUS,RP0		; bank 1
		bsf	PIE1,CMIE		; enables comparator int
		bcf	STATUS,RP0		; bank 0

		bcf	csig			; clears clap signal flag
		bcf	swsig			; clears switch flag
		return

;***** MAIN
Main		call	Init			; general settings
		sleep				; waits for ints
		nop

;** EXTERNAL SWITCH

Loop		btfss	swsig			; external switch
		goto 	Stop
		bcf	swsig			; clears external switch flag
		call	Relais			; switches relais

;** TOO MANY CLAPS

Stop		btfsc	led_flash		; LED flashing ?
		goto	Loop

;** 1ST CLAP

		btfsc	csig			; clap signal
		goto 	Clap			; starts again

;** NOTHING TO DO -> SLEEP

		movfw	Ton			; TMR1 not needed ?
		btfss	STATUS,Z
		goto	Loop
		sleep				; waits for ints
		goto	Loop

;** 1ST BREAK

Clap		bcf	csig			; clears clap signal flag
		movlw	d'15'			; sets postscaler for break
		movwf	MLX
		movfw	TMR0
		addlw	d'125'			; sets postscaler for break
		movwf	ML125
Wpause		btfsc	csig			; no signal for 500ms
		goto	Lbreak			; starts again
		movfw	ML125
		subwf	TMR0,w
		btfss	STATUS,Z
		goto	Wpause
		movlw	d'125'
		addwf	ML125,f
		decfsz	MLX,f
		goto	Wpause

;** 2ND CLAP

		movlw	d'30'			; sets postscaler for waiting for new signal
		movwf	MLX
WSig		btfsc	csig			; waits for second signal for 1s
		goto	Sig2
		movfw	ML125
		subwf	TMR0,w
		btfss	STATUS,Z
		goto	WSig
		movlw	d'125'
		addwf	ML125,f
		decfsz	MLX,f
		goto	WSig
		goto	Loop			; no second signal, starts again
Sig2		bcf	csig			; clears clap signal flag

;** 2ND BREAK

		movlw	d'15'			; sets postscaler for final break
		movwf	MLX
Fpause		btfsc	csig			; no signal for 500ms
		goto	Lbreak			; starts again
		movfw	ML125
		subwf	TMR0,w
		btfss	STATUS,Z
		goto	Fpause
		movlw	d'125'
		addwf	ML125,f
		decfsz	MLX,f
		goto	Fpause

;** CLAP SWITCH COUNTER
		
		incf	SCount,f		; increases switch counter

		bcf	T1CON,TMR1ON		; stops TMR1
		movlw	d'30'			; sets postscalers
		movwf	SWC30			; decreases switch counter every 30s
		movlw	d'200'			; sets switch counter postscaler
		movwf	SWC200
		btfsc	sco_w			; switch counter wait already set ?
		goto	SCcheck
		bsf	sco_w			; sets switch counter wait flag
		incf	Ton,f			; increases TMR1 on counter
SCcheck		bsf	T1CON,TMR1ON		; starts TMR1

		movlw	d'5'			; maximum claps in line without a pause of 30s
		subwf	SCount,w
		btfss	STATUS,C		; C is clear -> SCount < 5
		goto	Normal

;** TOO MANY CLAPS

		bcf	T1CON,TMR1ON		; stops TMR1
		movlw	d'240'			; sets postscaler
		movwf	LEDL			; LED is flashing for 60s
		movlw	d'50'			; sets postscaler
		movwf	LEDH
		movwf	LEDH_DEF		; sets LED postscaler default
		bsf	led_flash		; enables flashing
		btfsc	led_w			; LED already on ?
		goto	NeLoop
		bsf	LED			; enables LED
		bsf	led_w			; sets led wait flag
		incf	Ton,f			; increases TMR1 on counter
NeLoop		bsf	T1CON,TMR1ON		; starts TMR1	
		goto	Loop

;** SWITCH RELAIS

Normal		call	Relais			; switches relais
		goto	Loop			; starts again

;** SEQUENCE BREAK

Lbreak		bcf	csig			; clears clap signal flag
		goto	Loop			; starts again	

;***** INITIALIZE EEPROM LOCATIONS

EE		CODE	0x2100
		DE	0x00

		END     	                ; directive 'end of program'
